------------------------------------------------------------------------
-- Event:        Delphi Day 2018, Piacenza, June 06 2018               -
--               https://www.delphiday.it/                             -
-- Seminary:     How to write high performance queries in T-SQL        -
-- Demo:         Common Table Expression (CTE)                         -
-- Author:       Sergio Govoni                                         -
-- Notes:        --                                                    -
------------------------------------------------------------------------

USE [WideWorldImporters];
GO


-- Table Expression

-- Una Table Expression  una tabella derivata, completamente virtuale,
-- originata da una query

-- Una tabella derivata pu essere citata nella clausola FROM
-- come una qualsiasi altra tabella fisica

-- La visibilit di una tabella derivata  limitata
-- alla query esterna della CTE


-- Tabella derivata
SELECT
  t.a
  ,t.b
  ,t.c
  ,t.d
FROM
  (SELECT PersonID, FullName, EmailAddress, PhoneNumber
   FROM Application.People) AS t(a, b, c, d);
GO




-- Una Common Table Expression  un tipo di Table Expression

-- Una CTE si definisce attraverso l'utilizzo della clausola WITH
-- e di una query esterna che referenzia il nome della CTE

-- t is CTE name
-- (a, b, c, d) is CTE aliases

WITH CTE (a, b, c, d) AS
(
  -- Select di base
  SELECT PersonID, FullName, EmailAddress, PhoneNumber
  FROM Application.People
)
-- Outer query
SELECT
  a
  ,b
  ,c
  ,d
FROM
  CTE;
GO





-- CTE multiple

-- CTE non possono essere innestate, non  possibile definire una CTE
-- all'interno di un'altra CTE

-- E' comunque possibile definire CTE multiple utilizzando la stessa
-- clausola WITH

WITH C1 AS
(
  -- SELECT di base
  SELECT
    YEAR(oh.OrderDate) AS [Year]
    ,CustomerID AS idcli
  FROM
    Sales.Orders AS oh
),
C2 AS
(
  SELECT
    [Year]
    ,COUNT(DISTINCT idcli) AS num_active_customer
  FROM
    C1
  GROUP BY
    [Year]
)
SELECT
  *
FROM
  C2
WHERE
  num_active_customer >= 50;
GO



-- La stessa CTE pu essere ripetuta pi di una volta nella outer query
WITH ACTIVE_CLI AS
(
  -- Select di base
  SELECT
    YEAR(oh.OrderDate) AS [Year]
    ,COUNT(distinct CustomerID) AS active_customers
  FROM
    Sales.Orders AS oh
  GROUP BY
    YEAR(oh.OrderDate)
)
-- Select esterna
SELECT
  YCur.[Year]
  --,YPrv.anno
  ,YCur.active_customers as clienti_attivi_anno_corrente
  ,YPrv.active_customers as clienti_attivi_anno_precedente
  ,(YCur.active_customers - YPrv.active_customers) as diff
FROM
  ACTIVE_CLI AS YCur
LEFT OUTER JOIN
  ACTIVE_CLI AS YPrv ON (YCur.[Year] - 1 = YPrv.[Year])
ORDER BY
  [Year]
GO






-- Problema: Stampare le etichette prodotto finito per ogni gli
-- ordini di lavorazione con WorkOrderID 1 e 2

SELECT OrderID, OrderDate, CustomerID, PickingCompletedWhen FROM Sales.Orders WHERE OrderID = 48
SELECT OrderLineID, StockItemID, Quantity FROM Sales.OrderLines WHERE OrderID = 48
GO


-- CTE ricorsive
WITH cte AS
(
  -- Query di base
  SELECT
    CAST('Base query' AS VARCHAR) AS [row_type]
    ,YEAR(so.OrderDate) AS [Year]
	,sol.OrderLineID
    ,sol.Quantity
    ,1 AS label_number
  FROM
    Sales.Orders AS so
  JOIN
    Sales.OrderLines AS sol on so.OrderID=sol.OrderID
  WHERE
    so.OrderID = 48

  UNION ALL

  -- Query ricorsiva
  SELECT
    CAST('Recursive query' AS VARCHAR) AS tipo
    ,c.[Year] AS [Year]
    ,c.OrderLineID
    ,c.Quantity
    ,c.label_number + 1 AS label_number
  FROM
    cte AS c
  WHERE
    (c.label_number + 1 <= c.Quantity)
)
-- Query esterna
SELECT 
  *
FROM
  cte
ORDER BY
  cte.[Year], cte.OrderLineID, cte.label_number;
GO




-- Esercizio

/*
DROP INDEX idx_mgr_emp_i_fname_lname;
GO

CREATE UNIQUE INDEX idx_mgr_emp_i_fname_lname ON dbo.Employees
(
  mgrid
  ,empid
)
INCLUDE
(
  firstname
  ,lastname
);
GO
*/


SELECT * FROM Application.Employees
GO


-- Dato il dipendente con ID = 2, vogliamo venga restituito
-- lui stesso e tutti i suoi subordinati a tutti i livelli
WITH Emps AS
(
  -- Query di base "innesco" della ricorsione
  SELECT
    empid, mgrid, firstname, lastname
  FROM
    Application.Employees
  WHERE
    empid = 2

  UNION ALL

  -- Query ricorsiva
  -- Come fa questa query a diventare una query ricorsiva?
  -- Attraverso la referenza ricorsiva del nome della CTE "Emps" 
  -- La ricorsione di ferma quando la query ricorsiva restituisce un
  -- result-set vuoto

  -- Non c' un check esplicito per terminare la ricorsione
  SELECT
    Emp.empid, Emp.mgrid, Emp.firstname, Emp.lastname
  FROM /* FROM "result-set estratto nella query di base" */
    Emps AS Mgr
  JOIN
    Application.Employees AS Emp ON Emp.mgrid = Mgr.empid
)
-- Query esterna
SELECT
  *
FROM
  Emps
OPTION
  (MAXRECURSION 100 /*by default*/);
GO

/*
empid       mgrid       firstname  lastname
----------- ----------- ---------- --------------------
2           1           Don        Funk           | Query di base

3           2           Judy       Lew              | 1 ricorsione (mgr id = 2)
5           2           Sven       Buck             |

6           5           Paul       Suurs              | 2 ricorsione (mgr id = 5)
7           5           Russell    King               |
9           5           Zoya       Dolgopyatova       |

4           3           Yael       Peled              | 2 ricorsione (mgr id = 3)
8           3           Maria      Cameron            |

                                                        | 3 ricorsione = data-set vuoto
*/
